from abc import ABC, abstractmethod
import torch

import sys, os
python_path = os.path.dirname(os.path.dirname(__file__))
sys.path.append(python_path)


from torch_irl.format import default_preprocess_obss
from torch_irl.utils import DictList, ParallelEnv


class BaseAlgo(ABC):
    """The base class for RL algorithms."""

    def __init__(self, envs, acmodel, discmodel, device, num_frames_per_proc, discount, lr, gae_lambda, entropy_coef,
                 value_loss_coef, max_grad_norm, ac_recurrence, disc_recurrence, preprocess_obss, reshape_reward, entropy_reward):
        """
        Initializes a `BaseAlgo` instance.

        Parameters:
        ----------
        envs : list
            a list of environments that will be run in parallel
        acmodel : torch.Module
            the model
        num_frames_per_proc : int
            the number of frames collected by every process for an update
        discount : float
            the discount for future rewards
        lr : float
            the learning rate for optimizers
        gae_lambda : float
            the lambda coefficient in the GAE formula
            ([Schulman et al., 2015](https://arxiv.org/abs/1506.02438))
        entropy_coef : float
            the weight of the entropy cost in the final objective
        value_loss_coef : float
            the weight of the value loss in the final objective
        max_grad_norm : float
            gradient will be clipped to be at most this value
        recurrence : int
            the number of steps the gradient is propagated back in time
        preprocess_obss : function
            a function that takes observations returned by the environment
            and converts them into the format that the model can handle
        reshape_reward : function
            a function that shapes the reward, takes an
            (observation, action, reward, done) tuple as an input
        entropy_reward : bool
            whether adding entropy regularizatgion to the reward
            (observation, action, reward, done) tuple as an input
        """

        # Store parameters

        self.env = ParallelEnv(envs)
        self.acmodel = acmodel
        self.discmodel = discmodel
        self.device = device
        self.num_frames_per_proc = num_frames_per_proc
        self.discount = discount
        self.lr = lr
        self.gae_lambda = gae_lambda
        self.entropy_coef = entropy_coef
        self.value_loss_coef = value_loss_coef
        self.max_grad_norm = max_grad_norm
        self.ac_recurrence = ac_recurrence
        self.disc_recurrence = disc_recurrence
        self.preprocess_obss = default_preprocess_obss
        if preprocess_obss:
            self.preprocess_obss = preprocess_obss
            print("Customized observation preprocess")
        self.reshape_reward = reshape_reward
        self.entropy_reward = entropy_reward

        # Control parameters

        assert (self.acmodel.recurrent or self.ac_recurrence == 1) and (self.discmodel.recurrent or self.disc_recurrence == 1)
        assert self.num_frames_per_proc % self.ac_recurrence == 0 and self.num_frames_per_proc % self.disc_recurrence == 0

        # Configure acmodel

        self.acmodel.to(self.device)
        self.acmodel.train()
        self.discmodel.to(self.device)
        self.discmodel.train()
        # Store helpers values

        self.num_procs = len(envs)
        self.num_frames = self.num_frames_per_proc * self.num_procs
        
        self.num_demo_frames_per_proc = None
        self.num_demo_frames = None

        # Initialize experience values

        shape = (self.num_frames_per_proc, self.num_procs)

        self.obs = self.env.reset()
        self.obss = [None] * (shape[0])
        if self.acmodel.recurrent:
            self.ac_memory = torch.zeros(shape[1], self.acmodel.memory_size, device=self.device)
            self.ac_memories = torch.zeros(*shape, self.acmodel.memory_size, device=self.device)
        if self.discmodel.recurrent:
            self.disc_memory = torch.zeros(shape[1], self.discmodel.memory_size, device=self.device)
            self.disc_memories = torch.zeros(*shape, self.discmodel.memory_size, device=self.device)

        self.mask = torch.ones(shape[1], device=self.device)
        self.masks = torch.zeros(*shape, device=self.device)
        self.actions = torch.zeros(*shape, device=self.device, dtype=torch.int)
        self.values = torch.zeros(*shape, device=self.device)
        self.rewards = torch.zeros(*shape, device=self.device)
        self.ori_rewards = torch.zeros(*shape, device=self.device)
        self.advantages = torch.zeros(*shape, device=self.device)
        self.log_probs = torch.zeros(*shape, device=self.device)

        # Initialize log values

        self.log_episode_return = torch.zeros(self.num_procs, device=self.device)
        self.log_episode_reshaped_return = torch.zeros(self.num_procs, device=self.device)
        self.log_episode_num_frames = torch.zeros(self.num_procs, device=self.device)

        self.log_done_counter = 0
        self.log_return = [0] * self.num_procs
        self.log_reshaped_return = [0] * self.num_procs
        self.log_num_frames = [0] * self.num_procs

    def collect_demonstrations(self, demons_dict):
        demos = DictList()
        self.num_demo_frames_per_proc = len(demons_dict['obs'])
        self.num_demo_frames = self.num_demo_frames_per_proc * 1
         
        demos.obs = self.preprocess_obss(demons_dict['obs'], device=self.device)

        if self.acmodel.recurrent:
            demos.ac_memory = torch.zeros(len(demos.obs), *self.ac_memories.shape[2:]).to(self.device)
        if self.discmodel.recurrent:
            demos.disc_memory = torch.zeros(len(demos.obs), *self.disc_memories.shape[2:]).to(self.device)
        demos.mask = 1. - torch.tensor(demons_dict['done']).unsqueeze(1).unsqueeze(1).to(self.device)
        demos.action = torch.tensor(demons_dict['action']).to(self.device)
        
        return demos

    def collect_experiences(self):
        """Collects rollouts and computes advantages.

        Runs several environments concurrently. The next actions are computed
        in a batch mode for all environments at the same time. The rollouts
        and advantages from all environments are concatenated together.

        Returns
        -------
        exps : DictList
            Contains actions, rewards, advantages etc as attributes.
            Each attribute, e.g. `exps.reward` has a shape
            (self.num_frames_per_proc * num_envs, ...). k-th block
            of consecutive `self.num_frames_per_proc` frames contains
            data obtained from the k-th environment. Be careful not to mix
            data from different environments!
        logs : dict
            Useful stats about the training process, including the average
            reward, policy loss, value loss, etc.
        """

        for i in range(self.num_frames_per_proc):
            # Do one agent-environment interaction

            preprocessed_obs = self.preprocess_obss(self.obs, device=self.device)
            with torch.no_grad():
                if self.acmodel.recurrent:
                    dist, value, ac_memory = self.acmodel(preprocessed_obs, self.ac_memory * self.mask.unsqueeze(1))
                else:
                    dist, value = self.acmodel(preprocessed_obs)
            action = dist.sample()

            obs, ori_reward, done, info = self.env.step(action.cpu().numpy())
            with torch.no_grad():
                disc_output = self.discmodel(preprocessed_obs, action, self.disc_memory * self.mask.unsqueeze(1))
                disc_prob, disc_memory = disc_output[0], disc_output[-1]
            #disc_prob = disc_dist.log_prob(action).exp()
            reward = ((1 - disc_prob).log().flatten() - disc_prob.log().flatten()  + (dist.log_prob(action).flatten() if not self.entropy_reward else 0.)).detach().cpu().numpy().tolist()#
            # Update experiences values

            self.obss[i] = self.obs
            self.obs = obs
            if self.acmodel.recurrent:
                self.ac_memories[i] = self.ac_memory
                self.ac_memory = ac_memory
            if self.discmodel.recurrent:
                self.disc_memories[i] = self.disc_memory
                self.disc_memory = disc_memory
            self.masks[i] = self.mask
            self.mask = 1 - torch.tensor(done, device=self.device, dtype=torch.float)
            self.actions[i] = action
            self.values[i] = value
            if self.reshape_reward is not None:
                self.rewards[i] = torch.tensor([
                    self.reshape_reward(obs_, action_, reward_, done_)
                    for obs_, action_, reward_, done_ in zip(obs, action, reward, done)
                ], device=self.device)
                self.ori_rewards[i] = torch.tensor([
                    self.reshape_reward(obs_, action_, ori_reward_, done_)
                    for obs_, action_, ori_reward_, done_ in zip(obs, action, ori_reward, done)
                ], device=self.device)
            else:
                self.rewards[i] = torch.tensor(reward, device=self.device)
                self.ori_rewards[i] = torch.tensor(ori_reward, device=self.device)
            self.log_probs[i] = dist.log_prob(action)

            # Update log values

            self.log_episode_return += torch.tensor(ori_reward, device=self.device, dtype=torch.float)
            self.log_episode_reshaped_return += self.ori_rewards[i]
            self.log_episode_num_frames += torch.ones(self.num_procs, device=self.device)

            for i, done_ in enumerate(done):
                if done_:
                    self.log_done_counter += 1
                    self.log_return.append(self.log_episode_return[i].item())
                    self.log_reshaped_return.append(self.log_episode_reshaped_return[i].item())
                    self.log_num_frames.append(self.log_episode_num_frames[i].item())

            self.log_episode_return *= self.mask
            self.log_episode_reshaped_return *= self.mask
            self.log_episode_num_frames *= self.mask

        # Add advantage and return to experiences

        preprocessed_obs = self.preprocess_obss(self.obs, device=self.device)
        with torch.no_grad():
            if self.acmodel.recurrent:
                _, next_value, _ = self.acmodel(preprocessed_obs, self.ac_memory * self.mask.unsqueeze(1))
            else:
                _, next_value = self.acmodel(preprocessed_obs)

        for i in reversed(range(self.num_frames_per_proc)):
            next_mask = self.masks[i+1] if i < self.num_frames_per_proc - 1 else self.mask
            next_value = self.values[i+1] if i < self.num_frames_per_proc - 1 else next_value
            next_advantage = self.advantages[i+1] if i < self.num_frames_per_proc - 1 else 0

            delta = self.rewards[i] + self.discount * next_value * next_mask - self.values[i]
            self.advantages[i] = delta + self.discount * self.gae_lambda * next_advantage * next_mask

        # Define experiences:
        #   the whole experience is the concatenation of the experience
        #   of each process.
        # In comments below:
        #   - T is self.num_frames_per_proc,
        #   - P is self.num_procs,
        #   - D is the dimensionality.

        exps = DictList()
        exps.obs = [self.obss[i][j]
                    for j in range(self.num_procs)
                    for i in range(self.num_frames_per_proc)]
        if self.acmodel.recurrent:
            # T x P x D -> P x T x D -> (P * T) x D
            exps.ac_memory = self.ac_memories.transpose(0, 1).reshape(-1, *self.ac_memories.shape[2:])
        if self.discmodel.recurrent:
            # T x P x D -> P x T x D -> (P * T) x D
            exps.disc_memory = self.disc_memories.transpose(0, 1).reshape(-1, *self.disc_memories.shape[2:])
        # T x P -> P x T -> (P * T) x 1
        exps.mask = self.masks.transpose(0, 1).reshape(-1).unsqueeze(1)
        # for all tensors below, T x P -> P x T -> P * T
        exps.action = self.actions.transpose(0, 1).reshape(-1)
        exps.value = self.values.transpose(0, 1).reshape(-1)
        exps.reward = self.rewards.transpose(0, 1).reshape(-1)
        exps.ori_reward = self.ori_rewards.transpose(0, 1).reshape(-1)
        exps.advantage = self.advantages.transpose(0, 1).reshape(-1)
        exps.returnn = exps.value + exps.advantage
        exps.log_prob = self.log_probs.transpose(0, 1).reshape(-1)

        # Preprocess experiences

        exps.obs = self.preprocess_obss(exps.obs, device=self.device)

        # Log some values

        keep = max(self.log_done_counter, self.num_procs)

        logs = {
            "return_per_episode": self.log_return[-keep:],
            "reshaped_return_per_episode": self.log_reshaped_return[-keep:],
            "num_frames_per_episode": self.log_num_frames[-keep:],
            "num_frames": self.num_frames
        }

        self.log_done_counter = 0
        self.log_return = self.log_return[-self.num_procs:]
        self.log_reshaped_return = self.log_reshaped_return[-self.num_procs:]
        self.log_num_frames = self.log_num_frames[-self.num_procs:]

        return exps, logs

    @abstractmethod
    def update_ac_parameters(self):
        pass

    @abstractmethod
    def update_disc_parameters(self):
        pass
